/***************************************************************************/
/*                                                                         */
/*  idrv_stik.c                                                            */
/*                                                                         */
/*  Copyright (C) 2012 Monotype Imaging Inc. All rights reserved.          */
/*                                                                         */
/***************************************************************************/

#include "fs_api.h"

#ifdef FS_STIK

#define CROSS(A,B) (FixMul(A.x, B.y) - FixMul(A.y, B.x))
#define DOT(A,B) (FixMul(A.x,B.x) + FixMul(A.y,B.y))
#define SIN_2_DEGREES 2287 /* in 16.16 fixed of course */
#define SIN_5_DEGREES 5712
#define SIN_30_DEGREES 32768

typedef struct
{
    int num_contours;
    int num_types, max_types;
    int num_points, max_points;
    FS_BYTE *types;
    FS_FIXED *x, *y;
} SECTION;


/****************************************************************/
static FS_VOID set_bbox (_DS_ int n_pts, FS_OUTLINE *outl)
{
    FS_FIXED lo_x,hi_x,lo_y,hi_y, x, y;
    FS_FIXED *px, *py;
    int i;

    if (n_pts ==0 || outl==0)
        return;

    px = outl->x;
    lo_x = hi_x = px[0];
    py = outl->y;
    lo_y = hi_y = py[0];

    for (i=1; i<n_pts; i++)
    {
        x = px[i];
        if (x > hi_x) hi_x = x;
        if (x < lo_x) lo_x = x;
        y = py[i];
        if (y > hi_y) hi_y = y;
        if (y < lo_y) lo_y = y;
    }
    outl->lo_x = lo_x;
    outl->hi_x = hi_x;
    outl->lo_y = lo_y;
    outl->hi_y = hi_y;

    STATE.error = SUCCESS;
}

/****************************************************************/
/* find the intersection point <c> of given a point <x0,y0> with a
* unit tangent <p0>, with another point <x1,y1> with unit tangent <p1>
*
* assuming they intersect, for some <s> and <t> we have:
* x0 + s*p0.x == x1 + t*p1.x
* y0 + s*p0.y == y1 + t*p1.y
*
* re-arrange
* s*p0.x - t*p1.x == x1 - x0
* s*p0.y - t*p1.y == y1 - y0
*
* multiply
* s*p0.x*p1.y - t*p1.x*p1.y == p1.y*(x1 - x0)
* s*p0.y*p1.x - t*p1.y*p1.x == p1.x*(y1 - y0)
*
* subtract
* s*(p0.x * p1.y - p0.y * p1.x) == p1.y(x1 - x0) - p1.x(y1 - y0)
*
* divide
* s = (p1.y(x1 - x0) - p1.x(y1 - y0)) / (p0.x * p1.y - p0.y * p1.x)
*
* intersection point is therefore
* c->x = x0 + s*p0.x;
* c->y = y0 + s*p0.y;
*
* BUT OF COURSE THIS IS ALL HAPPENING IN FS_FIXED */

static int intersect(FS_FIXED x0,FS_FIXED y0,FIXED_VECTOR *p0,FS_FIXED x1,FS_FIXED y1,FIXED_VECTOR *p1,FIXED_VECTOR *c)
{
    FS_FIXED a,b,s;

    /* <p0> and <p1> are unit tangents ... <b> is the cross product which is the 
    * sin of the included angle which is in the invertal [-FIXED_ONE ... FIXED_ONE] */
    b = FixMul(p0->x, p1->y) - FixMul(p0->y, p1->x);

    /* unit tangents are almost parallel ... intersection will be bizarre
    * just use the midpoint of the chord from (x0,y0) to (x1,y1) */

    if (ABS(b) <= SIN_2_DEGREES)
    {
        c->x = (x0 + x1)/2;
        c->y = (y0 + y1)/2;
        return 1;
    }

    /* now we know we can subtract any two coordinates safely, and we're
     * multiplying by a value in the interval [-FIXED_ONE ... FIXED_ONE] so
     * there is no chance of overflow on <a> */
    a = FixMul(p1->y, x1 - x0) - FixMul(p1->x, y1 - y0);

    /* No assurances about overflow here ... BUT in the usage this was designed
     * for (finding the control point for the left/right offset parabola's of a 
     * stroke) we should be OK */

    s = FixDiv(a,b);
    c->x = x0 + FixMul(s,p0->x);
    c->y = y0 + FixMul(s,p0->y);

    /* return OK if (x,y) aren't at infinity */
    return (ABS(c->x)!=MYINFINITY && ABS(c->y)!=MYINFINITY) ? 0 : 1;
}


static FS_ULONG init_section(_DS_ SECTION *p, int np, int nt)
{
    SYS_MEMSET(p,0,sizeof(SECTION));
    p->max_types = nt;
    p->types = (FS_BYTE *)FSS_malloc(_PS_ nt);
    if (STATE.error) return STATE.error;
    p->max_points = np;
    p->x = (FS_FIXED *)FSS_malloc(_PS_ np * 4);
    if (STATE.error) return STATE.error;
    p->y = (FS_FIXED *)FSS_malloc(_PS_ np * 4);
    if (STATE.error) return STATE.error;
    return SUCCESS;
}

static FS_LONG add_M(_DS_ SECTION *p, FS_FIXED x, FS_FIXED y)
{
    p->num_contours++;
    if (p->num_types == p->max_types)
    {
        p->max_types += 32;
        p->types = (FS_BYTE *)FSS_realloc(_PS_ p->types, p->max_types);
        if (!p->types)
            return STATE.error;
    }
    if (p->num_points == p->max_points)
    {
        p->max_points += 64;
        p->x = (FS_FIXED *) FSS_realloc(_PS_ p->x, 4*p->max_points);
        p->y = (FS_FIXED *) FSS_realloc(_PS_ p->y, 4*p->max_points);
        if (!p->x || !p->y)
            return STATE.error;
    }
    p->types[p->num_types++] = FS_MOVETO;
    p->x[p->num_points] = x;
    p->y[p->num_points] = y;
    p->num_points++;

    return SUCCESS;
}

static FS_LONG add_L(_DS_ SECTION *p, FS_FIXED x, FS_FIXED y)
{
    if (p->num_types == p->max_types)
    {
        p->max_types += 32;
        p->types = (FS_BYTE *)FSS_realloc(_PS_ p->types, p->max_types);
        if (!p->types)
            return STATE.error;
    }
    if (p->num_points == p->max_points)
    {
        p->max_points += 64;
        p->x = (FS_FIXED *) FSS_realloc(_PS_ p->x, 4*p->max_points);
        p->y = (FS_FIXED *) FSS_realloc(_PS_ p->y, 4*p->max_points);
        if (!p->x || !p->y)
            return STATE.error;
    }
    p->types[p->num_types++] = FS_LINETO;
    p->x[p->num_points] = x;
    p->y[p->num_points] = y;
    p->num_points++;

    return SUCCESS;
}

static FS_LONG add_Q(_DS_ SECTION *p, FS_FIXED x1, FS_FIXED y1, FS_FIXED x2, FS_FIXED y2)
{
    if (p->num_types == p->max_types)
    {
        p->max_types += 32;
        p->types = (FS_BYTE *)FSS_realloc(_PS_ p->types, p->max_types);
        if (!p->types)
            return STATE.error;
    }
    if (p->num_points >= p->max_points - 1)
    {
        p->max_points += 64;
        p->x = (FS_FIXED *) FSS_realloc(_PS_ p->x, 4*p->max_points);
        p->y = (FS_FIXED *) FSS_realloc(_PS_ p->y, 4*p->max_points);
        if (!p->x || !p->y)
            return STATE.error;
    }
    p->types[p->num_types++] = FS_QUADTO;
    p->x[p->num_points] = x1;
    p->y[p->num_points] = y1;
    p->num_points++;
    p->x[p->num_points] = x2;
    p->y[p->num_points] = y2;
    p->num_points++;

    return SUCCESS;
}

/* append the elements of <right> to <left> BACKWARDS
*
* RIGHT
* 2.5   15        moveto
* 3     15
* 3     14.5    quadto
* 3     5.5        lineto
* <==> the segments
* quad(2.5 15) -> (3 15) -> (3 14.5)
* line(3 14.5) -> (3 5.5)
*
* LEFT
* 2.5   15        moveto
* 2     15
* 2     14.5    quadto
* 2     5.5        lineto
* 2     5
* 2.5   5        quadto
* 3     5
* 3     5.5        quadto
*
* want to reverse the segments of RIGHT, wrt
* both ORDER and DIRECTION.  In this case:
* line(3 5.5) -> (3 14.5)
* quad(3 14.5) -> (3 15) -> (2.5 15)
*
* or
*
* 3     14.5    lineto
* 3     15
* 2.5   15        quadto
*
* this can be done simply by processing the types in reverse
* (ignoring the initial moveto) and starting with the next
* to last <x,y> coordinate
*/
static FS_LONG copy_RL(_DS_ SECTION *right, SECTION *left)
{
    int i,j;
    FS_BYTE *types = right->types;
    FS_FIXED *x = right->x;
    FS_FIXED *y = right->y;

    for (i=right->num_types-1,j=right->num_points-2; i>=1; i--)
    {
        if (types[i]==FS_QUADTO)
        {
            add_Q(_PS_ left,x[j],y[j],x[j-1],y[j-1]);
            if (STATE.error) return STATE.error;
            j -= 2;
        }
        else if (types[i]==FS_LINETO)
        {
            add_L(_PS_ left,x[j],y[j]);
            if (STATE.error) return STATE.error;
            j--;
        }
        else
        {
            add_M(_PS_ left, x[j],y[j]);
            if (STATE.error) return STATE.error;
            j--;
        }
    }
    right->num_points = 0;
    right->num_types = 0;

    return SUCCESS;
}

/* start a contour with either a 1/2 circle or a 1/4 square end */
static FS_LONG start_contour(_DS_ SECTION *left, SECTION *right,
                             FS_FIXED x0, FS_FIXED y0,
                             FIXED_VECTOR *t1, FS_BOOLEAN round_end)
{
    FIXED_VECTOR n1;

    /* right hand normal */
    n1.y = -t1->x;
    n1.x = t1->y;

    if (round_end)
    {
        FS_FIXED sx,sy;

        /* start the contour with a half-round */
        sx = x0 - t1->x;
        sy = y0 - t1->y;

        /* right half */
        add_M(_PS_ right,sx,sy);
        if (STATE.error)
            return STATE.error;
        add_Q(_PS_ right,sx+n1.x,sy+n1.y,x0+n1.x,y0+n1.y);
        if (STATE.error)
            return STATE.error;

        /* left half */
        add_M(_PS_ left,sx,sy);    /* yes it is a duplicate... */
        if (STATE.error)
            return STATE.error;

        add_Q(_PS_ left,sx-n1.x,sy-n1.y,x0-n1.x,y0-n1.y);
        if (STATE.error)
            return STATE.error;
    }
    else
    {
        add_M(_PS_ right,x0-n1.x-t1->x/2, y0-n1.y-t1->y/2);
        if (STATE.error)
            return STATE.error;
        add_L(_PS_ right,x0+n1.x-t1->x/2,y0+n1.y-t1->y/2);
        if (STATE.error)
            return STATE.error;

        /* left half */
        add_M(_PS_ left, x0-n1.x-t1->x/2, y0-n1.y-t1->y/2);
        if (STATE.error)
            return STATE.error;
    }

    return STATE.error;
}

/* end the previous contour with either 1/2 circle or 1/4 square end */
static FS_LONG end_contour(_DS_ SECTION *left, SECTION *right, FS_FIXED sw, FS_BOOLEAN round_end)
{
    FIXED_VECTOR L;
    FIXED_VECTOR R,M,ut1,t1,n1;
    int n;

    /* the previous contour ends with... */
    n = left->num_points - 1;
    L.x = left->x[n];
    L.y = left->y[n];

    if (round_end)
    {
        n = right->num_points - 1;
        R.x = right->x[n];
        R.y = right->y[n];

        /* a midpoint */
        M.x = (L.x + R.x)/2;
        M.y = (L.y + R.y)/2;

        /* unit vector from R to L */
        ut1.x = L.x - R.x;
        ut1.y = L.y - R.y;
        fixed_norm(&ut1);

        /* the right hand normal points in the direction
           of the continuation of the curve...  */
        t1.x = FixMul(sw/2, ut1.x);
        t1.y = FixMul(sw/2, ut1.y);
        n1.y = -t1.x;
        n1.x = t1.y;

        /* add the 1/2 circle to close the contour */
        add_Q(_PS_ right, R.x + n1.x, R.y + n1.y, M.x + n1.x, M.y + n1.y);
        if (STATE.error)
            return STATE.error;
        add_Q(_PS_ right, L.x + n1.x, L.y + n1.y, L.x, L.y);
        if (STATE.error)
            return STATE.error;
    }
    else
    {
        /* line to close the contour */
        add_L(_PS_ right, L.x, L.y);
        if (STATE.error)
            return STATE.error;
    }

    /* now copy all the points of <right> onto <left> */
    copy_RL(_PS_ right,left);

    return STATE.error;
}

/**************************************************************************/
/* function to create a regular closed outline from open-ended strokes */
FS_OUTLINE *idrv_expand_stik(_DS_ FS_OUTLINE *stik, FS_LONG *_nt, FS_LONG *_np)
{
    FS_BOOLEAN need_to_free_stik = 0;
    FS_BOOLEAN round_end;
    FS_FIXED sw = STATE.stroke_pct * STATE.lpm;
    FS_FIXED x0,y0,x1,y1,x2,y2,xx0,yy0,xx2,yy2;
    int i,num, start = 1;    /* initialized to make the compiler happy */
    int nc,np,nt;
    FS_OUTLINE *outl = 0;
    FS_BYTE *types = stik->type;
    FS_FIXED *x = stik->x;
    FS_FIXED *y = stik->y;
    FS_FIXED sw_half;
    FS_FIXED xadj = 0;
    FS_FIXED yadj = 0;
    FS_FIXED diagonalsw;
    FS_FIXED orthogonalsw;
    FS_FIXED diagonalsw_half;
    FS_FIXED orthogonalsw_half;
    FS_FIXED fsw = MAX(FIXED_ONE,sw);
    FIXED_VECTOR L,R,M,PR,PL,ut0,ut1,t1,n1,ut2,t2,n2,pt;
    FIXED_VECTOR h_t1;
    SECTION left,right;

    sw = (fsw + FIXED_ONEHALF) & 0xFFFF0000;
    sw_half = sw/2;  /* integer width */
    round_end = 0; /* square end */

    diagonalsw = orthogonalsw = sw;

    if (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON)
    {
        if (sw < FIXED_ONE)
        {
            sw = FIXED_ONE;
            sw_half = FIXED_ONEHALF;
        }
        else
        {
            fsw = (3277 * STATE.lpm + FIXED_ONEHALF) & 0xFFFF0000;

            if (fsw & FIXED_ONE) 
            {   /* this is the odd case */
                xadj = (sw-FIXED_ONE)/2;
            }
            else
            {   /* this is the even case */
                xadj = sw_half-FIXED_ONE;
            }

            if (xadj > FIXED_ONEHALF)
                xadj = xadj-FIXED_ONE;
            if (-xadj > FIXED_ONEHALF)
                xadj = xadj+FIXED_ONE;

            yadj = xadj;
        }
        diagonalsw = FixMul(sw,55705);
        orthogonalsw = sw;
    }
    diagonalsw_half = diagonalsw/2;
    orthogonalsw_half = orthogonalsw/2;

    /* autohint non-offlined stroke fonts */
    if ( !(STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON))
    {
        FS_OUTLINE *ah_stik;
        /* stik passed in will be deleted by call chain */
        /* we need to free ah_stik when we are done with it */
        ah_stik = copy_outline(_PS_ stik, 0);
        if (ah_stik)
        {
            /* copy is successful */
            need_to_free_stik = 1;
            stik = ah_stik;
            types = stik->type;
            x = stik->x;
            y = stik->y;
        }
        autohint_stik(_PS_ stik, sw);
    }

    nc = stik->nc;

    *_np = *_nt = 0;    /* in preparation for failure ... */

    /* guess at parameters of expanded stik */
    np = (10*nc)+(3*stik->np)/2;
    nt = (5*nc)+(3*stik->num)/2;
    if (init_section(_PS_ &left,np,nt))
    {
        if (need_to_free_stik)
            FSS_free_char(_PS_ stik);
        return 0;
    }
    if (init_section(_PS_ &right,np,nt))
    {
        FSS_free(_PS_ left.x);
        FSS_free(_PS_ left.y);
        FSS_free(_PS_ left.types);
        if (need_to_free_stik)
            FSS_free_char(_PS_ stik);
        return 0;
    }

    num = stik->num;
    x0 = y0 = 0;    /* to remove a warning */
    PL.x = 0; PL.y = 0; PR.x = 0; PR.y = 0; ut0.x = 0; ut0.y = 0;
    for (i=0; i<num; i++)
    {
        FS_FIXED sin,cos;

        switch(types[i])
        {
        case FS_MOVETO:
            x0 = *x++;
            y0 = *y++;
            x0 += xadj;
            y0 += yadj;

            {
                 sw = orthogonalsw;
                 sw_half = orthogonalsw_half;
            }

            if (right.num_points)
            {
                end_contour(_PS_ &left, &right, sw, round_end);
                if (STATE.error)
                    goto fail;
            }
            start = 1;
            break;

        case FS_LINETO:
case_lineto:
             x1 = *x++;
             y1 = *y++;
             x1 += xadj;
             y1 += yadj;

             if (x1 == x0 || y1 == y0)
             {
                 sw = orthogonalsw;
                 sw_half = orthogonalsw_half;
             }
             else
             {
                 sw = diagonalsw;
                 sw_half = diagonalsw_half;
             }

             /* degenerate case, such as a period */
             if (x0==x1 && y0==y1)
             {
                if((i > 0) && (types[i-1] != FS_MOVETO))
                    continue; /* skip continuing line case */

                if (round_end)
                {
                    /* right half */
                    add_M(_PS_ &right,x0 + sw_half,y0);
                    if (STATE.error)
                        goto fail;
                    add_Q(_PS_ &right,x0 + sw_half, y0 + sw_half, x0, y0 + sw_half);
                    if (STATE.error)
                        goto fail;
                    add_Q(_PS_ &right,x0 - sw_half, y0 + sw_half, x0 - sw_half, y0);
                    if (STATE.error)
                        goto fail;

                    /* left half */
                    add_M(_PS_ &left,x0 - sw_half, y0);
                    if (STATE.error)
                        goto fail;
                    add_Q(_PS_ &left,x0 - sw_half, y0 - sw_half, x0, y0 - sw_half);
                    if (STATE.error)
                        goto fail;
                    add_Q(_PS_ &left,x0 + sw_half,y0 - sw_half, x0 + sw_half, y0);
                    if (STATE.error)
                        goto fail;
                }
                else
                {
                    /* add a square */
                    add_M(_PS_ &right,x0 + sw_half,y0 - sw_half);
                    if (STATE.error)
                        goto fail;

                    add_L(_PS_ &right,x0 + sw_half,y0 + sw_half);
                    if (STATE.error)
                        goto fail;

                    add_L(_PS_ &right,x0 - sw_half,y0 + sw_half);
                    if (STATE.error)
                        goto fail;

                    add_L(_PS_ &right,x0 - sw_half,y0 - sw_half);
                    if (STATE.error)
                        goto fail;

                    add_M(_PS_ &left,x0 - sw_half,y0 - sw_half);
                    if (STATE.error)
                        goto fail;

                    add_L(_PS_ &left,x0 + sw_half,y0 - sw_half); 
                    if (STATE.error)
                        goto fail;
                }

                /* save end point and tangent */
                /* unit tangent in direction of line */
                ut1.x = x1-x0;
                ut1.y = y1-y0;
                fixed_norm(&ut1);

                x0 = x1;
                y0 = y1;
                ut0 = ut1;
                break;
             }

             /* unit tangent in direction of line */
             ut1.x = x1-x0;
             ut1.y = y1-y0;
             fixed_norm(&ut1);

             /* tangent and normal of length sw/2 */
             t1.x = FixMul(sw_half, ut1.x);
             t1.y = FixMul(sw_half, ut1.y);
             n1.y = -t1.x;
             n1.x = t1.y;

             if (start)
             {
                 start_contour(_PS_ &left,&right,x0,y0,&t1, round_end);
                 if (STATE.error)
                     goto fail;

                 /* it IS a smooth joint */
                 ut0 = ut1;
                 /* no longer the start of a contour */
                 start = 0;
             }

             /* ? is the joint angular -- more than 5 degrees */
             sin = CROSS(ut0,ut1);
             cos = DOT(ut0,ut1);

             if (cos < 0) /* a sharp turn, more than 90 degrees    */
             {
                 FIXED_VECTOR t0,n0;

                 t0.x = FixMul(sw_half,ut0.x);
                 t0.y = FixMul(sw_half,ut0.y);
                 n0.y = -t0.x;
                 n0.x = t0.y;
                 L.x = x0 + t0.x - n0.x;
                 L.y = y0 + t0.y - n0.y;
                 R.x = x0 - t1.x - n1.x;
                 R.y = y0 - t1.y - n1.y;
                 M.x = (L.x + R.x)/2;
                 M.y = (L.y + R.y)/2;
                 add_Q(_PS_ &left,L.x,L.y,M.x,M.y);
                 add_Q(_PS_ &left,R.x,R.y,R.x+t1.x, R.y+t1.y);
                 add_L(_PS_ &right, x0+n1.x, y0+n1.y);
             }
             else if (sin > SIN_5_DEGREES)
             {
                 /* a left turn, causes gap on right */
                 L.x = x0-n1.x;
                 L.y = y0-n1.y;
                 R.x = x0+n1.x;
                 R.y = y0+n1.y;

                 intersect(PL.x, PL.y,&ut0, L.x, L.y,&ut1, &pt);
                 left.x[left.num_points - 1] = pt.x;
                 left.y[left.num_points - 1] = pt.y;
                 add_L(_PS_ &left, pt.x, pt.y);
                 if (STATE.error)
                     goto fail;

                 /* find the intersection point of the right-hand
                     vectors (PR + ut0) and (R + ut1) */
                 intersect(PR.x, PR.y, &ut0,R.x, R.y, &ut1, &pt);

                 /* span the gap with a little parabola */
                 add_Q(_PS_ &right, pt.x, pt.y, R.x, R.y);
                 if (STATE.error)
                        goto fail;
             }
             else if (sin < -SIN_5_DEGREES)
             {
                 /* a right turn, causes gap on left */
                 L.x = x0-n1.x;
                 L.y = y0-n1.y;
                 R.x = x0+n1.x;
                 R.y = y0+n1.y;

                 /* find the intersection point of the left-hand
                    vectors (PL + ut0) and (L + ut1) */
                 intersect(PL.x, PL.y, &ut0, L.x, L.y, &ut1, &pt);
                 /* span the gap with a little parabola */
                 add_Q(_PS_ &left, pt.x, pt.y, L.x, L.y);
                 if (STATE.error)
                     goto fail;

                 intersect(PR.x, PR.y,&ut0, R.x, R.y, &ut1, &pt);
                 right.x[right.num_points - 1] = pt.x;
                 right.y[right.num_points - 1] = pt.y;
                 add_L(_PS_ &right, pt.x, pt.y);
                 if (STATE.error)
                     goto fail;
             }

             /* now that gaps are filled ... do the left/right lines */
             if (round_end)
             {
                 add_L(_PS_ &right,PR.x = x1+n1.x, PR.y = y1+n1.y);
                 if (STATE.error) 
                     goto fail;
                 add_L(_PS_ &left,PL.x = x1-n1.x, PL.y = y1-n1.y);
                 if (STATE.error)
                     goto fail;
             }
             else
             {
                 h_t1.x = FixMul(sw/4, ut1.x);
                 h_t1.y = FixMul(sw/4, ut1.y);
                 add_L(_PS_ &right,PR.x = x1+n1.x+h_t1.x, PR.y = y1+n1.y+h_t1.y);
                 if (STATE.error)
                     goto fail;
                 add_L(_PS_ &left,PL.x = x1-n1.x+h_t1.x, PL.y = y1-n1.y+h_t1.y);
                 if (STATE.error)
                     goto fail;
             }

             /* save end point and tangent */
             x0 = x1;
             y0 = y1;
             ut0 = ut1;
             break;

        case FS_QUADTO:
             x1 = *x++;
             y1 = *y++;
             x2 = *x++;
             y2 = *y++;
             x1 += xadj;
             y1 += yadj;
             x2 += xadj;
             y2 += yadj;

             {
                 sw = diagonalsw;
                 sw_half = diagonalsw_half;
             }

             /* degenerate cases */
             if ((x0==x1 && y0==y1) || (x1==x2 && y1==y2))
             {
                 *--x = x2;
                 *--y = y2;
                 goto case_lineto;
             }

             /* unit tangent at <x0,y0> */
             ut1.x = x1-x0;
             ut1.y = y1-y0;
             fixed_norm(&ut1);

             /* tangent and normal of length sw/2 */
             t1.x = FixMul(sw_half, ut1.x);
             t1.y = FixMul(sw_half, ut1.y);
             n1.y = -t1.x;
             n1.x = t1.y;

             /* unit tangent at <x2,y2> */
             ut2.x = x2-x1;
             ut2.y = y2-y1;
             fixed_norm(&ut2);

             /* check and fix poorly hinted cases. sin is the angle between the legs */
             /* of the quad. if almost flat ... use a line instead                   */
             sin = CROSS(ut1,ut2);
             if ( ABS(sin) < SIN_30_DEGREES &&
                  ((ut1.x&0x80000000)!=(ut2.x&0x80000000)) &&
                  ((ut1.y&0x80000000)!=(ut2.y&0x80000000)) )
             {
                *--x = x2;
                *--y = y2;
                goto case_lineto;
             }

             /* tangent and normal of length sw/2 */
             t2.x = FixMul(sw_half, ut2.x);
             t2.y = FixMul(sw_half, ut2.y);
             n2.y = -t2.x;
             n2.x = t2.y;

             if (start)
             {
                 start_contour(_PS_ &left,&right,x0,y0,&t1, round_end);
                 if (STATE.error)
                     goto fail;

                 /* it IS a smooth joint */
                 ut0 = ut1;

                 /* no longer the start of a contour */
                 start = 0;
             }

             /* ? is the joint angular -- more than 5 degrees */
             sin = CROSS(ut0,ut1);
             cos = DOT(ut0,ut1);
             if (cos < 0)
             {
                 FIXED_VECTOR t0,n0;

                 t0.x = FixMul(sw_half,ut0.x);
                 t0.y = FixMul(sw_half,ut0.y);
                 n0.y = -t0.x;
                 n0.x = t0.y;
                 L.x = x0 + t0.x - n0.x;
                 L.y = y0 + t0.y - n0.y;
                 R.x = x0 - t1.x - n1.x;
                 R.y = y0 - t1.y - n1.y;
                 M.x = (L.x + R.x)/2;
                 M.y = (L.y + R.y)/2;
                 add_Q(_PS_ &left,L.x,L.y,M.x,M.y);
                 add_Q(_PS_ &left,R.x,R.y,R.x+t1.x, R.y+t1.y);
                 add_L(_PS_ &right, x0+n1.x, y0+n1.y);
             }
             else if (sin > SIN_5_DEGREES)
             {
                 /* a left turn, causes gap on right */
                 L.x = x0-n1.x;
                 L.y = y0-n1.y;
                 R.x = x0+n1.x;
                 R.y = y0+n1.y;

                 intersect(PL.x, PL.y,&ut0, L.x, L.y,&ut1, &pt);
                 left.x[left.num_points - 1] = pt.x;
                 left.y[left.num_points - 1] = pt.y;
                 add_L(_PS_ &left, pt.x, pt.y);
                 if (STATE.error)
                     goto fail;

                 /* find the intersection point of the right-hand
                    vectors (PR + ut0) and (R + ut1) */
                 intersect(PR.x, PR.y, &ut0,R.x, R.y, &ut1, &pt);

                 /* span the gap with a little parabola */
                 add_Q(_PS_ &right, pt.x, pt.y, R.x, R.y);
                 if (STATE.error)
                     goto fail;
             }
             else if (sin < -SIN_5_DEGREES)
             {
                 /* a right turn, causes gap on left */
                 L.x = x0-n1.x;
                 L.y = y0-n1.y;
                 R.x = x0+n1.x;
                 R.y = y0+n1.y;

                 /* find the intersection point of the left-hand
                    vectors (PL + ut0) and (L + ut1) */
                 intersect(PL.x, PL.y, &ut0, L.x, L.y, &ut1, &pt);

                 /* span the gap with a little parabola */
                 add_Q(_PS_ &left, pt.x, pt.y, L.x, L.y);
                 if (STATE.error)
                     goto fail;

                 intersect(PR.x, PR.y,&ut0, R.x, R.y,&ut1, &pt);
                 right.x[right.num_points - 1] = pt.x;
                 right.y[right.num_points - 1] = pt.y;
                 add_L(_PS_ &right, pt.x, pt.y);
                 if (STATE.error)
                     goto fail;
             }

             /* now that any gaps have been filled, do the left/right quads
                start point of the new left parabola */
             xx0 = x0 - n1.x;
             yy0 = y0 - n1.y;
             /* endpoint of the new left parabola */
             xx2 = x2 - n2.x;
             yy2 = y2 - n2.y;
             /* control point is at intersection of control arms */
             intersect(xx0,yy0,&t1,xx2,yy2,&t2,&pt);

             /* add the left quad */
             add_Q(_PS_ &left, pt.x, pt.y, PL.x = xx2, PL.y = yy2);
             if (STATE.error)
                 goto fail;

             /* start point of the new right parabola */
             xx0 = x0 + n1.x;
             yy0 = y0 + n1.y;
             /* end point of the new right parabola */
             xx2 = x2 + n2.x;
             yy2 = y2 + n2.y;
             /* control point is at intersection of control arms */
             intersect(xx0,yy0,&t1,xx2,yy2,&t2,&pt);

             /* add the right quad */
             add_Q(_PS_ &right, pt.x, pt.y, PR.x = xx2, PR.y = yy2);
             if (STATE.error)
                 goto fail;

             /* save final point and tangent */
             x0 = x2;
             y0 = y2;
             ut0 = ut2;

             break;

        default:
             /* there is no default. */
             goto fail;
        } /* switch (types[i]) */

    } /* for (i=0; i<num; i++) */

    /* finish the last contour */
    if (right.num_points)
    {
        end_contour(_PS_ &left, &right, sw, round_end);
        if (STATE.error)
            goto fail;
    }

    /* almost done ... now make the mess into an FS_OUTLINE */
    outl = setup_outline(_PS_ (FS_SHORT)left.num_contours, left.num_types, left.num_points);
    if (outl)
    {
        FS_BYTE *t = outl->type;
        FS_FIXED *px = outl->x;
        FS_FIXED *py = outl->y;

        /* move the data into an FS_OUTLINE */
        if (t)
        {
            SYS_MEMCPY(t, left.types, left.num_types);
            /* it is now an outline char */
            outl->type[0] |= OUTLINE_CHAR;
        }
        if (px) SYS_MEMCPY(px, left.x, left.num_points * 4);
        if (py) SYS_MEMCPY(py, left.y, left.num_points * 4);
        set_bbox(_PS_ left.num_points, outl);
        *_np = left.num_points;
        *_nt = left.num_types;
        outl->i_dx = stik->i_dx;
        outl->i_dy = stik->i_dy;
        outl->dx = stik->dx;
        outl->dy = stik->dy;
    }

    fail:

    /* free temps */
    FSS_free(_PS_ right.x);
    FSS_free(_PS_ right.y);
    FSS_free(_PS_ right.types);
    FSS_free(_PS_ left.x);
    FSS_free(_PS_ left.y);
    FSS_free(_PS_ left.types);
    if (need_to_free_stik)
        FSS_free_char(_PS_ stik);
    return outl;
}
#endif
